/*****************************************************************************
 * libmp4.c: libmp4 tests
 *****************************************************************************
 * Copyright (C) 2026 VideoLabs, VLC authors and VideoLAN
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston MA 02110-1301, USA.
 *****************************************************************************/
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <vlc/vlc.h>
#include "../../../lib/libvlc_internal.h"
#include "../../libvlc/test.h"

#include <vlc_common.h>
#include <vlc_block.h>
#include <vlc_stream.h>

#define MODULE_STRING "test_demux_libmp4"

#include "../../../modules/demux/mp4/libmp4.h"

#define EXPECT(a) do {\
    if(!(a)) { \
        fprintf(stderr, "failed line %d\n", __LINE__); \
        goto failed; } \
    } while(0)

const char vlc_module_name[] = MODULE_STRING;

static MP4_Box_t * BoxAddChild( MP4_Box_t *p_parent, MP4_Box_t *p_childbox )
{
    if( !p_childbox )
        return NULL;
    if( !p_parent->p_first )
            p_parent->p_first = p_childbox;
    else
            p_parent->p_last->p_next = p_childbox;
    p_parent->p_last = p_childbox;
    p_childbox->p_father = p_parent;
    return p_childbox;
}

static MP4_Box_t *newBox(const char x[4])
{
    return MP4_BoxNew(VLC_FOURCC(x[0],x[1],x[2],x[3]));
}

static MP4_Box_t *generate_mp4_tree()
{
    MP4_Box_t *root = NULL, *moov, *trak, *mdia, *stbl;

    if (!(root = newBox("root")))
        goto failed;

    if (!BoxAddChild(root, newBox("ftyp")))
        goto failed;

    if (!(moov = BoxAddChild(root, newBox("moov"))))
        goto failed;

    if (!(trak = BoxAddChild(moov, newBox("trak"))))
        goto failed;

    if (!(mdia = BoxAddChild(trak, newBox("mdia"))))
        goto failed;

    if (!(mdia = BoxAddChild(mdia, newBox("minf"))))
        goto failed;

    if (!(stbl = BoxAddChild(mdia, newBox("stbl"))))
        goto failed;

    if (!BoxAddChild(stbl, newBox("stsd")))
        goto failed;
    if (!BoxAddChild(stbl, newBox("stts")))
        goto failed;
    if (!BoxAddChild(stbl, newBox("stsz")))
        goto failed;
    if (!BoxAddChild(stbl, newBox("stco")))
        goto failed;

    if (!(trak = BoxAddChild(moov, newBox("trak"))))
        goto failed;
    trak->i_pos = 1;

    BoxAddChild(root, newBox("mdat"));

    return root;

failed:
    MP4_BoxFree(root);
    return NULL;
}

static int check_MP4_BoxGet()
{
    MP4_Box_t *extracted = NULL;
    MP4_Box_t *root = generate_mp4_tree();
    if(!root)
      return 1;

    MP4_Box_t *box;
    EXPECT(MP4_BoxGet(root, ".") == root);
    EXPECT(MP4_BoxGet(NULL, ".") == NULL);

    box = MP4_BoxGet(root, "/moov");
    EXPECT(box);
    EXPECT(box->i_type == ATOM_moov);
    EXPECT(MP4_BoxGet(root, "moov") == box);

    box = MP4_BoxGet(root, "moov/trak/mdia/minf/stbl/stsd");
    EXPECT(box);
    EXPECT(box->i_type == ATOM_stsd);

    box = MP4_BoxGet(box, ".././stts");
    EXPECT(box);
    EXPECT(box->i_type == ATOM_stts);

    box = MP4_BoxGet(root, "moov/trak[0]/mdia/minf/stbl/stsd");
    EXPECT(box);
    EXPECT(box->i_type == ATOM_stsd);

    box = MP4_BoxGet(root, "moov/trak[1]/mdia/minf/stbl/stsd");
    EXPECT(!box);
    box = MP4_BoxGet(root, "moov/trak[1]");
    EXPECT(box);
    EXPECT(box->i_type == ATOM_trak);
    EXPECT(box->i_pos == 1);

    EXPECT(MP4_BoxCount(root, "moov/zzzz") == 0);
    EXPECT(MP4_BoxCount(root, "moov") == 1);
    EXPECT(MP4_BoxCount(root, "moov/trak") == 2);

    box = MP4_BoxGet(root, "ftyp");
    EXPECT(box);
    extracted = MP4_BoxExtract(&box, ATOM_mdat);
    EXPECT(extracted);
    EXPECT(extracted->i_type == ATOM_mdat);
    EXPECT(MP4_BoxCount(extracted, ".") == 1);
    MP4_BoxFree(extracted);

    MP4_BoxFree(root);

    return 0;

failed:
    MP4_BoxFree(root);
    MP4_BoxFree(extracted);
    return 1;
}

/*
+ ftyp size 28 offset 0
+ free size 8 offset 28
+ mdat size 1556 offset 36
+ moov size 794 offset 1592
|   + mvhd size 108 offset 1600
|   + trak size 581 offset 1708
|   |   + tkhd size 92 offset 1716
|   |   + edts size 36 offset 1808
|   |   |   + elst size 28 offset 1816
|   |   + mdia size 445 offset 1844
|   |   |   + mdhd size 32 offset 1852
|   |   |   + hdlr size 45 offset 1884
|   |   |   + minf size 360 offset 1929
|   |   |   |   + smhd size 16 offset 1937
|   |   |   |   + dinf size 36 offset 1953
|   |   |   |   |   + dref size 28 offset 1961
|   |   |   |   |   |   + url  size 12 offset 1977
|   |   |   |   + stbl size 300 offset 1989
|   |   |   |   |   + stsd size 126 offset 1997
|   |   |   |   |   |   + mp4a size 110 offset 2013
|   |   |   |   |   |   |   + esds size 54 offset 2049
|   |   |   |   |   |   |   + btrt size 20 offset 2103
|   |   |   |   |   + stts size 32 offset 2123
|   |   |   |   |   + stsc size 28 offset 2155
|   |   |   |   |   + stsz size 32 offset 2183
|   |   |   |   |   + stco size 20 offset 2215
|   |   |   |   |   + sgpd size 26 offset 2235
|   |   |   |   |   + sbgp size 28 offset 2261
|   + udta size 97 offset 2289
|   |   + meta size 89 offset 2297
|   |   |   + hdlr size 33 offset 2309
|   |   |   + ilst size 44 offset 2342
|   |   |   |   + ctoo size 36 offset 2350
|   |   |   |   |   + data size 28 offset 2358
*/

static const unsigned char mp4_slowstart[] = {
  0x00, 0x00, 0x00, 0x1c, 0x66, 0x74, 0x79, 0x70, 0x69, 0x73, 0x6f, 0x6d,
  0x00, 0x00, 0x02, 0x00, 0x69, 0x73, 0x6f, 0x6d, 0x69, 0x73, 0x6f, 0x32,
  0x6d, 0x70, 0x34, 0x31, 0x00, 0x00, 0x00, 0x08, 0x66, 0x72, 0x65, 0x65,
  0x00, 0x00, 0x06, 0x14, 0x6d, 0x64, 0x61, 0x74, 0xde, 0x02, 0x00, 0x4c,
  0x61, 0x76, 0x63, 0x36, 0x31, 0x2e, 0x31, 0x39, 0x2e, 0x31, 0x30, 0x31,
  0x00, 0x02, 0x30, 0xa8, 0x59, 0x59, 0x70, 0x4b, 0x1c, 0x29, 0x87, 0x41,
  0x41, 0x58, 0x5e, 0x35, 0x79, 0x51, 0xbd, 0x69, 0x64, 0x93, 0x7c, 0xe5,
  0xdd, 0xe5, 0xdd, 0x71, 0xdc, 0xda, 0x6b, 0x71, 0xaf, 0xff, 0x8c, 0x7e,
  0x84, 0x85, 0xd7, 0xb2, 0x95, 0x10, 0x0a, 0xd8, 0x64, 0x0e, 0xca, 0x24,
  0x3f, 0x4f, 0x43, 0x21, 0x8b, 0x07, 0xba, 0xc3, 0x43, 0x83, 0x60, 0x55,
  0x72, 0x05, 0x57, 0x54, 0x76, 0x5a, 0x27, 0x5b, 0x81, 0xd6, 0x71, 0xd7,
  0x72, 0x36, 0x4c, 0x80, 0x9d, 0x62, 0x5c, 0x96, 0x75, 0x9d, 0xfa, 0x87,
  0xa9, 0x62, 0x5c, 0x8c, 0x14, 0xf3, 0x5f, 0x11, 0xb0, 0xf3, 0x1e, 0xcc,
  0xfa, 0x64, 0xbf, 0xb4, 0xf3, 0xaa, 0x4b, 0x96, 0x79, 0x17, 0xfe, 0xb2,
  0xeb, 0x22, 0x5c, 0xb3, 0xc9, 0x62, 0x5c, 0xac, 0xb3, 0xa4, 0x66, 0x7e,
  0x37, 0x59, 0x70, 0xf9, 0x52, 0xad, 0x55, 0x6a, 0xa6, 0xfc, 0x15, 0x0c,
  0x5a, 0x55, 0xae, 0xac, 0x19, 0x65, 0xd5, 0xb6, 0xd8, 0xc5, 0x27, 0x3f,
  0xc8, 0xa8, 0x62, 0x47, 0x71, 0xb4, 0xc4, 0x97, 0x3c, 0x71, 0xf2, 0x9d,
  0x94, 0x40, 0x2a, 0x20, 0xb0, 0x65, 0x77, 0x93, 0x8d, 0x03, 0x21, 0x28,
  0x85, 0x0b, 0xc4, 0x32, 0xd9, 0x32, 0x18, 0xab, 0xe4, 0x2a, 0x4a, 0x21,
  0x09, 0x84, 0x0c, 0x1f, 0xda, 0x7e, 0x03, 0xfc, 0x9f, 0x31, 0xa3, 0x39,
  0x83, 0xea, 0x1b, 0x33, 0x08, 0x58, 0x68, 0x48, 0x71, 0x3b, 0x2b, 0x3e,
  0xee, 0xbe, 0xa0, 0x2d, 0xb5, 0xf6, 0x2b, 0x16, 0x73, 0x08, 0x8b, 0xbd,
  0xbd, 0xba, 0x06, 0xa0, 0xc3, 0x06, 0x75, 0x0b, 0x46, 0xc1, 0x97, 0x7e,
  0x16, 0xe7, 0xd3, 0xf6, 0xca, 0xbb, 0xdb, 0xed, 0xcb, 0x6d, 0x50, 0xe5,
  0xe3, 0x8d, 0xfc, 0x84, 0x4b, 0x6d, 0xba, 0xb1, 0xe5, 0xcf, 0x39, 0x03,
  0x05, 0x33, 0xb6, 0x81, 0x90, 0x91, 0xdc, 0x3c, 0xc1, 0xe6, 0x53, 0x4d,
  0xb5, 0x00, 0x09, 0x9d, 0xc0, 0x1c, 0x8a, 0x8a, 0x8a, 0x89, 0x8e, 0x61,
  0x82, 0x13, 0x71, 0xc0, 0x34, 0x21, 0x2c, 0x68, 0x31, 0x31, 0xc8, 0x02,
  0x81, 0x00, 0x5b, 0x9c, 0x03, 0x00, 0x01, 0xcb, 0x9e, 0x96, 0xd2, 0xd3,
  0x0f, 0xcf, 0xf6, 0xfe, 0xdf, 0x77, 0xbb, 0xab, 0x9e, 0x77, 0x19, 0xb2,
  0xdb, 0x18, 0xdc, 0x37, 0x39, 0x6d, 0x9c, 0x0b, 0x26, 0xb3, 0xc7, 0x3e,
  0x82, 0x37, 0x24, 0x01, 0x04, 0x19, 0x05, 0xc9, 0x16, 0x67, 0x99, 0x2e,
  0x79, 0xad, 0x99, 0xe6, 0x50, 0x99, 0xae, 0x76, 0xce, 0xd9, 0xe7, 0x28,
  0x2a, 0xa8, 0x2a, 0xe6, 0xb9, 0xaa, 0x94, 0x41, 0xa9, 0x65, 0x96, 0x51,
  0x38, 0x42, 0x80, 0x60, 0xe0, 0x0e, 0x39, 0x11, 0xe4, 0xe4, 0xec, 0xf9,
  0xdf, 0x63, 0xf4, 0x7f, 0xbf, 0xf3, 0xfa, 0xb7, 0xb6, 0x7f, 0x73, 0x58,
  0xd8, 0xf1, 0xe3, 0x28, 0x44, 0x33, 0xc0, 0xd5, 0xff, 0x6f, 0x82, 0xc1,
  0xe1, 0xff, 0xd7, 0xfe, 0x3e, 0xb4, 0x9f, 0x30, 0xd2, 0x0f, 0xab, 0xc3,
  0xfa, 0x7f, 0x8f, 0x86, 0x37, 0xaf, 0x2f, 0xb7, 0xab, 0xfe, 0xdf, 0xcb,
  0x14, 0xdb, 0x69, 0x1b, 0x6c, 0x9f, 0x1c, 0xe5, 0xa4, 0x4c, 0x72, 0x7d,
  0xa7, 0xbf, 0x12, 0xbb, 0xa7, 0x20, 0x4b, 0x64, 0x39, 0xf7, 0x1b, 0x27,
  0x8d, 0xbe, 0x43, 0x99, 0x74, 0xd9, 0x66, 0xa9, 0x0d, 0xf5, 0x02, 0x7d,
  0x1f, 0xa2, 0x10, 0xc1, 0x71, 0xd2, 0x18, 0xec, 0xf9, 0x1e, 0x3f, 0x81,
  0x27, 0xe4, 0x7f, 0x31, 0x93, 0xe8, 0x7c, 0x04, 0x9f, 0x61, 0xe9, 0xa4,
  0xeb, 0xe8, 0xc9, 0xf8, 0x23, 0xbd, 0x93, 0xc3, 0x60, 0x89, 0xf7, 0x0e,
  0xde, 0x4a, 0x5a, 0xc9, 0xf3, 0x4e, 0x8c, 0x48, 0xf0, 0x49, 0xf1, 0x8e,
  0x0e, 0x43, 0x5f, 0xc0, 0x89, 0xe9, 0x78, 0x19, 0x0c, 0x56, 0x54, 0x9f,
  0x0e, 0xe7, 0xc4, 0xf4, 0xbb, 0x52, 0x7d, 0x5b, 0x8c, 0x90, 0xcd, 0xe1,
  0x09, 0xf2, 0xde, 0x78, 0x4e, 0x3e, 0x00, 0x9f, 0x57, 0xe8, 0x73, 0xfb,
  0x49, 0xf3, 0xce, 0x0c, 0x43, 0x89, 0xf1, 0xd2, 0x78, 0xdc, 0x49, 0x0e,
  0x39, 0xc1, 0xf8, 0x01, 0x28, 0x9e, 0xda, 0x89, 0xfe, 0x2c, 0xd6, 0x95,
  0x5a, 0xb9, 0x95, 0x4e, 0x4d, 0x1b, 0x35, 0xf8, 0x6f, 0xce, 0xda, 0xdf,
  0xed, 0xbf, 0xde, 0x75, 0xef, 0x9f, 0xe3, 0x9f, 0x6f, 0x6e, 0x6f, 0xad,
  0x7f, 0x7f, 0xe9, 0xff, 0x77, 0x5c, 0xef, 0xf7, 0xff, 0xe3, 0xef, 0xf0,
  0xab, 0xe3, 0xf6, 0xff, 0x8e, 0x3f, 0xeb, 0xae, 0x7b, 0xfe, 0x7f, 0xd7,
  0xe3, 0xeb, 0xd5, 0xbc, 0xef, 0xf6, 0xff, 0xa7, 0xfe, 0xbc, 0x5f, 0xcf,
  0xf9, 0x1b, 0x95, 0x57, 0x6e, 0x51, 0xd2, 0xad, 0x52, 0xf5, 0x47, 0x6d,
  0xa5, 0x27, 0x1e, 0xbc, 0xf7, 0x80, 0xb1, 0x15, 0xa4, 0x89, 0x75, 0x46,
  0x5b, 0xab, 0x3c, 0xae, 0xf9, 0xd6, 0xa4, 0x14, 0xd1, 0xbc, 0x50, 0x3b,
  0x8d, 0xaf, 0x50, 0xa6, 0x23, 0x0c, 0x6c, 0xf1, 0xe7, 0x21, 0x73, 0x88,
  0xc3, 0x2a, 0xab, 0xf2, 0x13, 0xc6, 0x2b, 0x55, 0xea, 0xac, 0x7a, 0xaf,
  0x43, 0xc6, 0x30, 0xd4, 0x94, 0x77, 0x1d, 0x07, 0xec, 0x38, 0x5f, 0x13,
  0xe2, 0xf2, 0xbc, 0xfb, 0x5f, 0x29, 0x2e, 0x87, 0xe4, 0x90, 0xdd, 0xe8,
  0xfb, 0x5a, 0x11, 0x4a, 0x32, 0x23, 0x57, 0x75, 0x06, 0x36, 0x62, 0xba,
  0xb2, 0x87, 0x87, 0xbc, 0xec, 0xd8, 0x93, 0xc5, 0xa4, 0xb7, 0x42, 0xe8,
  0xf1, 0x94, 0x80, 0x28, 0x85, 0x2d, 0xeb, 0x7c, 0x96, 0x22, 0xed, 0xb2,
  0x36, 0xa6, 0xfe, 0xab, 0x36, 0x63, 0x59, 0x4d, 0xf6, 0xbb, 0xaa, 0x64,
  0xe5, 0x1c, 0xe9, 0x65, 0xbb, 0xf5, 0x6e, 0x40, 0x3c, 0xe0, 0x5d, 0xa4,
  0xf3, 0x4c, 0xfc, 0xc3, 0xb0, 0x17, 0x07, 0x10, 0xe8, 0xe6, 0xfa, 0x48,
  0xa0, 0x82, 0x70, 0x68, 0x9c, 0x3f, 0xe7, 0x49, 0x74, 0x7b, 0xa4, 0x51,
  0x45, 0x00, 0x79, 0x81, 0x41, 0x84, 0x72, 0x1b, 0xe3, 0xee, 0x85, 0xa5,
  0x0d, 0x11, 0xba, 0x8c, 0x28, 0x6f, 0x3a, 0xe9, 0xaf, 0x15, 0x4b, 0x5f,
  0x53, 0x22, 0xac, 0x7c, 0x72, 0xeb, 0x7b, 0x4d, 0xa5, 0x5b, 0x43, 0x32,
  0x46, 0xb5, 0x52, 0xc1, 0xb1, 0x5b, 0x34, 0x79, 0x7c, 0x1a, 0xd8, 0x07,
  0xc7, 0xb2, 0x24, 0xe7, 0x2d, 0x90, 0x93, 0x8c, 0x31, 0x35, 0xc9, 0x0b,
  0xc2, 0xbc, 0x50, 0x17, 0xe5, 0x63, 0x1e, 0xe9, 0x29, 0x5f, 0x71, 0x18,
  0xb2, 0x74, 0x72, 0x36, 0x4d, 0xd4, 0x6a, 0xec, 0x40, 0x25, 0xfe, 0x12,
  0x03, 0x31, 0x98, 0x9e, 0x01, 0x4b, 0x6b, 0xce, 0x79, 0x36, 0x19, 0x7d,
  0xcb, 0x80, 0xbd, 0xdf, 0x16, 0xb1, 0xa0, 0xa7, 0xe8, 0x1a, 0x33, 0x16,
  0x71, 0xaf, 0xeb, 0x13, 0x63, 0xfb, 0x64, 0x04, 0x9d, 0xfd, 0x62, 0x66,
  0xd3, 0x84, 0x8e, 0x28, 0x86, 0x7c, 0x1e, 0x07, 0x6f, 0x5b, 0xb0, 0x5e,
  0xf1, 0xd8, 0x36, 0x91, 0x48, 0xb6, 0x21, 0x85, 0x4b, 0xfd, 0xd5, 0x86,
  0x02, 0xb9, 0x2e, 0x58, 0x6c, 0x37, 0xee, 0x36, 0x60, 0x51, 0x04, 0x21,
  0xa3, 0x99, 0x1c, 0xe7, 0x2d, 0xa6, 0x3b, 0xbe, 0x3b, 0x8c, 0xcc, 0x5e,
  0x58, 0x75, 0x1f, 0xa0, 0xec, 0xce, 0xe2, 0xf1, 0xd3, 0xf9, 0x47, 0xe2,
  0x74, 0x32, 0xa9, 0xb4, 0xe1, 0x1a, 0xbf, 0xa7, 0xa2, 0xef, 0xdc, 0x62,
  0xe0, 0xc4, 0xd6, 0x7d, 0x89, 0x7d, 0x64, 0xf9, 0x8d, 0xaf, 0x2f, 0xe4,
  0x9f, 0x9b, 0xa4, 0x36, 0x62, 0xba, 0xda, 0x31, 0xe6, 0xa0, 0x57, 0x9a,
  0x49, 0x7a, 0xad, 0x40, 0x7c, 0x9d, 0x3a, 0x13, 0xa2, 0x37, 0x10, 0x37,
  0xd0, 0xe8, 0x27, 0x37, 0x1c, 0x1d, 0xd0, 0xfa, 0x44, 0x1c, 0x51, 0x45,
  0xa7, 0xd3, 0x5b, 0x59, 0x78, 0x59, 0x55, 0x65, 0xc9, 0xb6, 0x95, 0xb6,
  0xec, 0x61, 0x23, 0xa6, 0x3d, 0xfb, 0x27, 0xe2, 0x73, 0xed, 0x00, 0x38,
  0xf0, 0x8e, 0x3a, 0x24, 0x99, 0x6b, 0x5d, 0xfb, 0xa2, 0x66, 0xcf, 0x30,
  0xf6, 0x99, 0x7c, 0x01, 0x18, 0x9f, 0xb6, 0xe3, 0x2b, 0xe6, 0x0d, 0xfe,
  0x43, 0xf2, 0x7d, 0xdf, 0x69, 0xf5, 0x3f, 0x27, 0xdc, 0x7e, 0x23, 0x7a,
  0x15, 0xf3, 0x9f, 0xab, 0xf1, 0x82, 0x4e, 0x27, 0x28, 0x58, 0x56, 0x60,
  0x2d, 0xd0, 0xf0, 0x17, 0xf4, 0xe4, 0xfa, 0x48, 0x2b, 0xbd, 0x94, 0x0d,
  0xf6, 0x87, 0xf0, 0x1d, 0xe3, 0x07, 0x90, 0x1d, 0x65, 0x78, 0x3e, 0x64,
  0x88, 0x87, 0xd3, 0xdf, 0x28, 0x0c, 0x63, 0xe2, 0x52, 0xf7, 0x0c, 0x17,
  0x33, 0xec, 0x37, 0xe3, 0xfe, 0x71, 0x06, 0x59, 0x08, 0x54, 0xf7, 0xee,
  0x9b, 0x77, 0xfa, 0xfc, 0xf8, 0xd8, 0x69, 0xd5, 0x3b, 0xcd, 0xc9, 0x9f,
  0x5f, 0x18, 0x5c, 0x01, 0xff, 0x21, 0x55, 0x04, 0x49, 0x28, 0x8e, 0x03,
  0x12, 0x4f, 0x1d, 0x32, 0x67, 0x2f, 0x8d, 0x62, 0x5f, 0xd0, 0xc7, 0xd6,
  0x6b, 0xd9, 0x0b, 0xb1, 0xbe, 0xc3, 0x8a, 0x39, 0xdc, 0x08, 0x48, 0x43,
  0x4f, 0x58, 0xbe, 0xc5, 0xad, 0xe1, 0x4a, 0xb5, 0x6c, 0xf6, 0xfd, 0x91,
  0xe7, 0x1c, 0x3a, 0xa1, 0x91, 0xdc, 0x44, 0x58, 0x26, 0x81, 0x4c, 0x33,
  0x8d, 0xb1, 0xe9, 0xc9, 0xb4, 0xd6, 0x80, 0xe5, 0x81, 0xda, 0x63, 0xdb,
  0xc9, 0xdf, 0x11, 0x7b, 0xfd, 0x5a, 0xdf, 0xea, 0x62, 0x0a, 0x80, 0xfc,
  0x79, 0x8e, 0x2f, 0xd4, 0xbe, 0x77, 0x05, 0x70, 0xdc, 0x7c, 0x6f, 0xbc,
  0x77, 0xef, 0xde, 0xeb, 0x43, 0x11, 0x13, 0x98, 0xf7, 0xe3, 0xf2, 0x5b,
  0x3b, 0xff, 0x05, 0x0d, 0x0a, 0x0b, 0x1c, 0x15, 0x73, 0xcb, 0x71, 0x7a,
  0xc7, 0x67, 0x3a, 0xdd, 0x10, 0xe4, 0xb4, 0xd6, 0x53, 0x72, 0x34, 0x1d,
  0xec, 0xb9, 0x2b, 0x60, 0x68, 0xb8, 0x34, 0xd9, 0x24, 0xd1, 0xf7, 0x02,
  0xc6, 0x1a, 0x04, 0xe5, 0x62, 0x3c, 0xc8, 0x88, 0x0d, 0x40, 0x83, 0x7a,
  0xf3, 0xef, 0x12, 0xa8, 0x7f, 0xbf, 0x53, 0xbd, 0xb7, 0x2b, 0x32, 0xad,
  0x19, 0x4f, 0x5b, 0xa2, 0xb5, 0x02, 0x00, 0x00, 0x48, 0xe9, 0x93, 0xd2,
  0x00, 0x01, 0x2a, 0xa8, 0x91, 0xd2, 0x00, 0x01, 0xdf, 0xf5, 0x28, 0x00,
  0x00, 0x30, 0x63, 0xca, 0xe3, 0x00, 0x00, 0x22, 0x42, 0x56, 0x80, 0x00,
  0x00, 0xc7, 0xc6, 0xc1, 0x50, 0x00, 0x01, 0x2e, 0x10, 0x91, 0x8f, 0x6f,
  0x84, 0x8a, 0xca, 0x00, 0x00, 0x00, 0x01, 0x83, 0x36, 0x83, 0x49, 0x17,
  0x3a, 0xd0, 0x58, 0x00, 0x00, 0x00, 0x07, 0xb1, 0x5a, 0x22, 0xfd, 0x65,
  0x9c, 0x90, 0x00, 0x00, 0x00, 0x02, 0x20, 0x2c, 0xa0, 0x5a, 0x0c, 0x77,
  0x69, 0x00, 0x00, 0x00, 0x00, 0x09, 0x51, 0xc6, 0x93, 0x89, 0x1c, 0x8d,
  0x35, 0x92, 0x0c, 0x2a, 0xc9, 0xe4, 0x55, 0x08, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x80, 0x88, 0x4a, 0x01, 0xf2, 0x0b, 0xa5, 0x45, 0x92, 0x6a,
  0xe5, 0x14, 0xcb, 0x27, 0x24, 0x11, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x13, 0x99, 0x10, 0x8e, 0x2b, 0x01, 0x83, 0x61, 0x2c, 0x6c,
  0x11, 0x1a, 0xb6, 0xc8, 0x39, 0xa4, 0xaa, 0xce, 0xae, 0x68, 0x11, 0xac,
  0x42, 0x52, 0x2b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x13, 0x41, 0xc8, 0xc8, 0x6e, 0x76, 0x89, 0x41, 0xb4, 0x8b, 0x13,
  0x50, 0x07, 0x02, 0x2c, 0x9a, 0x0a, 0x24, 0x1e, 0xec, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xc0, 0x00, 0x00, 0x03, 0x1a,
  0x6d, 0x6f, 0x6f, 0x76, 0x00, 0x00, 0x00, 0x6c, 0x6d, 0x76, 0x68, 0x64,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x03, 0xe8, 0x00, 0x00, 0x01, 0x01, 0x00, 0x01, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x45, 0x74, 0x72, 0x61, 0x6b,
  0x00, 0x00, 0x00, 0x5c, 0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00,
  0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24,
  0x65, 0x64, 0x74, 0x73, 0x00, 0x00, 0x00, 0x1c, 0x65, 0x6c, 0x73, 0x74,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x00,
  0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbd,
  0x6d, 0x64, 0x69, 0x61, 0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68, 0x64,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0xb6, 0x00, 0x00, 0x0b, 0x5a, 0x55, 0xc4, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x2d, 0x68, 0x64, 0x6c, 0x72, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x73, 0x6f, 0x75, 0x6e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x6f, 0x75, 0x6e,
  0x64, 0x48, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x01,
  0x68, 0x6d, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00, 0x10, 0x73, 0x6d, 0x68,
  0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x24, 0x64, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65,
  0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01,
  0x2c, 0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00, 0x7e, 0x73, 0x74, 0x73,
  0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x6e, 0x6d, 0x70, 0x34, 0x61, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
  0x10, 0x00, 0x00, 0x00, 0x00, 0x1c, 0xb6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x65, 0x73, 0x64, 0x73, 0x00, 0x00, 0x00, 0x00, 0x03, 0x80, 0x80,
  0x80, 0x25, 0x00, 0x01, 0x00, 0x04, 0x80, 0x80, 0x80, 0x17, 0x40, 0x15,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xac, 0x44, 0x00, 0x00, 0x7a, 0x5a, 0x05,
  0x80, 0x80, 0x80, 0x05, 0x16, 0x08, 0x56, 0xe5, 0x00, 0x06, 0x80, 0x80,
  0x80, 0x01, 0x02, 0x00, 0x00, 0x00, 0x14, 0x62, 0x74, 0x72, 0x74, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xac, 0x44, 0x00, 0x00, 0x7a, 0x5a, 0x00,
  0x00, 0x00, 0x20, 0x73, 0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x04, 0x00, 0x00,
  0x00, 0x00, 0x01, 0x00, 0x00, 0x03, 0x5a, 0x00, 0x00, 0x00, 0x1c, 0x73,
  0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
  0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00,
  0x00, 0x00, 0x20, 0x73, 0x74, 0x73, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x02, 0x3b, 0x00,
  0x00, 0x02, 0x04, 0x00, 0x00, 0x01, 0xcd, 0x00, 0x00, 0x00, 0x14, 0x73,
  0x74, 0x63, 0x6f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
  0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0x1a, 0x73, 0x67, 0x70, 0x64, 0x01,
  0x00, 0x00, 0x00, 0x72, 0x6f, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x02, 0x00,
  0x00, 0x00, 0x01, 0xff, 0xff, 0x00, 0x00, 0x00, 0x1c, 0x73, 0x62, 0x67,
  0x70, 0x00, 0x00, 0x00, 0x00, 0x72, 0x6f, 0x6c, 0x6c, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x61, 0x75, 0x64, 0x74, 0x61, 0x00, 0x00, 0x00, 0x59, 0x6d, 0x65, 0x74,
  0x61, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x21, 0x68, 0x64, 0x6c,
  0x72, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6d, 0x64, 0x69,
  0x72, 0x61, 0x70, 0x70, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x2c, 0x69, 0x6c, 0x73, 0x74, 0x00, 0x00,
  0x00, 0x24, 0xa9, 0x74, 0x6f, 0x6f, 0x00, 0x00, 0x00, 0x1c, 0x64, 0x61,
  0x74, 0x61, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x4c, 0x61,
  0x76, 0x66, 0x36, 0x31, 0x2e, 0x37, 0x2e, 0x31, 0x30, 0x30
};
static const unsigned int mp4_slowstart_len = 2386;

static int check_MP4_Box(vlc_object_t *obj)
{
    stream_t *stream = vlc_stream_MemoryNew(obj, (uint8_t *)mp4_slowstart, mp4_slowstart_len, true);
    if(!stream)
        return 1;
    MP4_Box_t *root = MP4_BoxGetRoot(stream);
    vlc_stream_Delete(stream);

    EXPECT(root);

    MP4_BoxDumpStructure(obj, root);

    const MP4_Box_t *box = MP4_BoxGet(root, "moov/trak/mdia/minf/stbl/stsd");
    EXPECT(box);
    EXPECT(box->i_type == ATOM_stsd);

    MP4_BoxFree(root);
    return 0;

failed:
    MP4_BoxFree(root);
    return 1;
}

/*
+ ftyp size 24 offset 0
+ moov size 700 offset 24
|   + mvhd size 108 offset 32
|   + trak size 447 offset 140
|   |   + tkhd size 92 offset 148
|   |   + mdia size 347 offset 240
|   |   |   + mdhd size 32 offset 248
|   |   |   + hdlr size 45 offset 280
|   |   |   + minf size 262 offset 325
|   |   |   |   + smhd size 16 offset 333
|   |   |   |   + dinf size 36 offset 349
|   |   |   |   |   + dref size 28 offset 357
|   |   |   |   |   |   + url  size 12 offset 373
|   |   |   |   + stbl size 202 offset 385
|   |   |   |   |   + stsd size 126 offset 393
|   |   |   |   |   |   + mp4a size 110 offset 409
|   |   |   |   |   |   |   + esds size 54 offset 445
|   |   |   |   |   |   |   + btrt size 20 offset 499
|   |   |   |   |   + stts size 16 offset 519
|   |   |   |   |   + stsc size 16 offset 535
|   |   |   |   |   + stsz size 20 offset 551
|   |   |   |   |   + stco size 16 offset 571
|   + mvex size 40 offset 587
|   |   + trex size 32 offset 595
|   + udta size 97 offset 627
|   |   + meta size 89 offset 635
|   |   |   + hdlr size 33 offset 647
|   |   |   + ilst size 44 offset 680
|   |   |   |   + ctoo size 36 offset 688
|   |   |   |   |   + data size 28 offset 696
*/

static const unsigned char mp4_frag[] = {
  0x00, 0x00, 0x00, 0x18, 0x66, 0x74, 0x79, 0x70, 0x69, 0x73, 0x6f, 0x36,
  0x00, 0x00, 0x02, 0x00, 0x69, 0x73, 0x6f, 0x36, 0x6d, 0x70, 0x34, 0x31,
  0x00, 0x00, 0x02, 0xbc, 0x6d, 0x6f, 0x6f, 0x76, 0x00, 0x00, 0x00, 0x6c,
  0x6d, 0x76, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x01, 0xbf,
  0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c, 0x74, 0x6b, 0x68, 0x64,
  0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
  0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x01, 0x5b, 0x6d, 0x64, 0x69, 0x61, 0x00, 0x00, 0x00, 0x20,
  0x6d, 0x64, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0xb6, 0x00, 0x00, 0x00, 0x00,
  0x55, 0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x68, 0x64, 0x6c, 0x72,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x73, 0x6f, 0x75, 0x6e,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x53, 0x6f, 0x75, 0x6e, 0x64, 0x48, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72,
  0x00, 0x00, 0x00, 0x01, 0x06, 0x6d, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
  0x10, 0x73, 0x6d, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
  0x1c, 0x64, 0x72, 0x65, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0xca, 0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00,
  0x7e, 0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x6e, 0x6d, 0x70, 0x34, 0x61, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x01, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x1c, 0xb6, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x36, 0x65, 0x73, 0x64, 0x73, 0x00, 0x00, 0x00,
  0x00, 0x03, 0x80, 0x80, 0x80, 0x25, 0x00, 0x01, 0x00, 0x04, 0x80, 0x80,
  0x80, 0x17, 0x40, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0xac, 0x44, 0x00,
  0x00, 0xac, 0x44, 0x05, 0x80, 0x80, 0x80, 0x05, 0x16, 0x08, 0x56, 0xe5,
  0x00, 0x06, 0x80, 0x80, 0x80, 0x01, 0x02, 0x00, 0x00, 0x00, 0x14, 0x62,
  0x74, 0x72, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xac, 0x44, 0x00,
  0x00, 0xac, 0x44, 0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x74, 0x73, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x73,
  0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x14, 0x73, 0x74, 0x73, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x73,
  0x74, 0x63, 0x6f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x28, 0x6d, 0x76, 0x65, 0x78, 0x00, 0x00, 0x00, 0x20, 0x74,
  0x72, 0x65, 0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
  0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x61, 0x75, 0x64, 0x74, 0x61, 0x00,
  0x00, 0x00, 0x59, 0x6d, 0x65, 0x74, 0x61, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x21, 0x68, 0x64, 0x6c, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x6d, 0x64, 0x69, 0x72, 0x61, 0x70, 0x70, 0x6c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2c,
  0x69, 0x6c, 0x73, 0x74, 0x00, 0x00, 0x00, 0x24, 0xa9, 0x74, 0x6f, 0x6f,
  0x00, 0x00, 0x00, 0x1c, 0x64, 0x61, 0x74, 0x61, 0x00, 0x00, 0x00, 0x01,
  0x00, 0x00, 0x00, 0x00, 0x4c, 0x61, 0x76, 0x66, 0x36, 0x31, 0x2e, 0x37,
  0x2e, 0x31, 0x30, 0x30, 0x00, 0x00, 0x00, 0x64, 0x6d, 0x6f, 0x6f, 0x66,
  0x00, 0x00, 0x00, 0x10, 0x6d, 0x66, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x4c, 0x74, 0x72, 0x61, 0x66,
  0x00, 0x00, 0x00, 0x1c, 0x74, 0x66, 0x68, 0x64, 0x00, 0x02, 0x00, 0x38,
  0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x02, 0x3b,
  0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x74, 0x66, 0x64, 0x74,
  0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x14, 0x74, 0x72, 0x75, 0x6e, 0x01, 0x00, 0x00, 0x01,
  0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x02, 0x43,
  0x6d, 0x64, 0x61, 0x74, 0xde, 0x02, 0x00, 0x4c, 0x61, 0x76, 0x63, 0x36,
  0x31, 0x2e, 0x31, 0x39, 0x2e, 0x31, 0x30, 0x31, 0x00, 0x02, 0x30, 0xa8,
  0x59, 0x59, 0x70, 0x4b, 0x1c, 0x29, 0x87, 0x41, 0x41, 0x58, 0x5e, 0x35,
  0x79, 0x51, 0xbd, 0x69, 0x64, 0x93, 0x7c, 0xe5, 0xdd, 0xe5, 0xdd, 0x71,
  0xdc, 0xda, 0x6b, 0x71, 0xaf, 0xff, 0x8c, 0x7e, 0x84, 0x85, 0xd7, 0xb2,
  0x95, 0x10, 0x0a, 0xd8, 0x64, 0x0e, 0xca, 0x24, 0x3f, 0x4f, 0x43, 0x21,
  0x8b, 0x07, 0xba, 0xc3, 0x43, 0x83, 0x60, 0x55, 0x72, 0x05, 0x57, 0x54,
  0x76, 0x5a, 0x27, 0x5b, 0x81, 0xd6, 0x71, 0xd7, 0x72, 0x36, 0x4c, 0x80,
  0x9d, 0x62, 0x5c, 0x96, 0x75, 0x9d, 0xfa, 0x87, 0xa9, 0x62, 0x5c, 0x8c,
  0x14, 0xf3, 0x5f, 0x11, 0xb0, 0xf3, 0x1e, 0xcc, 0xfa, 0x64, 0xbf, 0xb4,
  0xf3, 0xaa, 0x4b, 0x96, 0x79, 0x17, 0xfe, 0xb2, 0xeb, 0x22, 0x5c, 0xb3,
  0xc9, 0x62, 0x5c, 0xac, 0xb3, 0xa4, 0x66, 0x7e, 0x37, 0x59, 0x70, 0xf9,
  0x52, 0xad, 0x55, 0x6a, 0xa6, 0xfc, 0x15, 0x0c, 0x5a, 0x55, 0xae, 0xac,
  0x19, 0x65, 0xd5, 0xb6, 0xd8, 0xc5, 0x27, 0x3f, 0xc8, 0xa8, 0x62, 0x47,
  0x71, 0xb4, 0xc4, 0x97, 0x3c, 0x71, 0xf2, 0x9d, 0x94, 0x40, 0x2a, 0x20,
  0xb0, 0x65, 0x77, 0x93, 0x8d, 0x03, 0x21, 0x28, 0x85, 0x0b, 0xc4, 0x32,
  0xd9, 0x32, 0x18, 0xab, 0xe4, 0x2a, 0x4a, 0x21, 0x09, 0x84, 0x0c, 0x1f,
  0xda, 0x7e, 0x03, 0xfc, 0x9f, 0x31, 0xa3, 0x39, 0x83, 0xea, 0x1b, 0x33,
  0x08, 0x58, 0x68, 0x48, 0x71, 0x3b, 0x2b, 0x3e, 0xee, 0xbe, 0xa0, 0x2d,
  0xb5, 0xf6, 0x2b, 0x16, 0x73, 0x08, 0x8b, 0xbd, 0xbd, 0xba, 0x06, 0xa0,
  0xc3, 0x06, 0x75, 0x0b, 0x46, 0xc1, 0x97, 0x7e, 0x16, 0xe7, 0xd3, 0xf6,
  0xca, 0xbb, 0xdb, 0xed, 0xcb, 0x6d, 0x50, 0xe5, 0xe3, 0x8d, 0xfc, 0x84,
  0x4b, 0x6d, 0xba, 0xb1, 0xe5, 0xcf, 0x39, 0x03, 0x05, 0x33, 0xb6, 0x81,
  0x90, 0x91, 0xdc, 0x3c, 0xc1, 0xe6, 0x53, 0x4d, 0xb5, 0x00, 0x09, 0x9d,
  0xc0, 0x1c, 0x8a, 0x8a, 0x8a, 0x89, 0x8e, 0x61, 0x82, 0x13, 0x71, 0xc0,
  0x34, 0x21, 0x2c, 0x68, 0x31, 0x31, 0xc8, 0x02, 0x81, 0x00, 0x5b, 0x9c,
  0x03, 0x00, 0x01, 0xcb, 0x9e, 0x96, 0xd2, 0xd3, 0x0f, 0xcf, 0xf6, 0xfe,
  0xdf, 0x77, 0xbb, 0xab, 0x9e, 0x77, 0x19, 0xb2, 0xdb, 0x18, 0xdc, 0x37,
  0x39, 0x6d, 0x9c, 0x0b, 0x26, 0xb3, 0xc7, 0x3e, 0x82, 0x37, 0x24, 0x01,
  0x04, 0x19, 0x05, 0xc9, 0x16, 0x67, 0x99, 0x2e, 0x79, 0xad, 0x99, 0xe6,
  0x50, 0x99, 0xae, 0x76, 0xce, 0xd9, 0xe7, 0x28, 0x2a, 0xa8, 0x2a, 0xe6,
  0xb9, 0xaa, 0x94, 0x41, 0xa9, 0x65, 0x96, 0x51, 0x38, 0x42, 0x80, 0x60,
  0xe0, 0x0e, 0x39, 0x11, 0xe4, 0xe4, 0xec, 0xf9, 0xdf, 0x63, 0xf4, 0x7f,
  0xbf, 0xf3, 0xfa, 0xb7, 0xb6, 0x7f, 0x73, 0x58, 0xd8, 0xf1, 0xe3, 0x28,
  0x44, 0x33, 0xc0, 0xd5, 0xff, 0x6f, 0x82, 0xc1, 0xe1, 0xff, 0xd7, 0xfe,
  0x3e, 0xb4, 0x9f, 0x30, 0xd2, 0x0f, 0xab, 0xc3, 0xfa, 0x7f, 0x8f, 0x86,
  0x37, 0xaf, 0x2f, 0xb7, 0xab, 0xfe, 0xdf, 0xcb, 0x14, 0xdb, 0x69, 0x1b,
  0x6c, 0x9f, 0x1c, 0xe5, 0xa4, 0x4c, 0x72, 0x7d, 0xa7, 0xbf, 0x12, 0xbb,
  0xa7, 0x20, 0x4b, 0x64, 0x39, 0xf7, 0x1b, 0x27, 0x8d, 0xbe, 0x43, 0x99,
  0x74, 0xd9, 0x66, 0xa9, 0x0d, 0xf5, 0x02, 0x7d, 0x1f, 0xa2, 0x10, 0xc1,
  0x71, 0xd2, 0x18, 0xec, 0xf9, 0x1e, 0x3f, 0x81, 0x27, 0xe4, 0x7f, 0x31,
  0x93, 0xe8, 0x7c, 0x04, 0x9f, 0x61, 0xe9, 0xa4, 0xeb, 0xe8, 0xc9, 0xf8,
  0x23, 0xbd, 0x93, 0xc3, 0x60, 0x89, 0xf7, 0x0e, 0xde, 0x4a, 0x5a, 0xc9,
  0xf3, 0x4e, 0x8c, 0x48, 0xf0, 0x49, 0xf1, 0x8e, 0x0e, 0x43, 0x5f, 0xc0,
  0x89, 0xe9, 0x78, 0x19, 0x0c, 0x56, 0x54, 0x9f, 0x0e, 0xe7, 0xc4, 0xf4,
  0xbb, 0x52, 0x7d, 0x5b, 0x8c, 0x90, 0xcd, 0xe1, 0x09, 0xf2, 0xde, 0x78,
  0x4e, 0x3e, 0x00, 0x9f, 0x57, 0xe8, 0x73, 0xfb, 0x49, 0xf3, 0xce, 0x0c,
  0x43, 0x89, 0xf1, 0xd2, 0x78, 0xdc, 0x49, 0x0e, 0x39, 0xc1, 0xf8, 0x00,
  0x00, 0x00, 0x64, 0x6d, 0x6f, 0x6f, 0x66, 0x00, 0x00, 0x00, 0x10, 0x6d,
  0x66, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
  0x00, 0x00, 0x4c, 0x74, 0x72, 0x61, 0x66, 0x00, 0x00, 0x00, 0x1c, 0x74,
  0x66, 0x68, 0x64, 0x00, 0x02, 0x00, 0x38, 0x00, 0x00, 0x00, 0x01, 0x00,
  0x00, 0x04, 0x00, 0x00, 0x00, 0x02, 0x04, 0x02, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x14, 0x74, 0x66, 0x64, 0x74, 0x01, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x14, 0x74,
  0x72, 0x75, 0x6e, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
  0x00, 0x00, 0x6c, 0x00, 0x00, 0x02, 0x0c, 0x6d, 0x64, 0x61, 0x74, 0x01,
  0x28, 0x9e, 0xda, 0x89, 0xfe, 0x2c, 0xd6, 0x95, 0x5a, 0xb9, 0x95, 0x4e,
  0x4d, 0x1b, 0x35, 0xf8, 0x6f, 0xce, 0xda, 0xdf, 0xed, 0xbf, 0xde, 0x75,
  0xef, 0x9f, 0xe3, 0x9f, 0x6f, 0x6e, 0x6f, 0xad, 0x7f, 0x7f, 0xe9, 0xff,
  0x77, 0x5c, 0xef, 0xf7, 0xff, 0xe3, 0xef, 0xf0, 0xab, 0xe3, 0xf6, 0xff,
  0x8e, 0x3f, 0xeb, 0xae, 0x7b, 0xfe, 0x7f, 0xd7, 0xe3, 0xeb, 0xd5, 0xbc,
  0xef, 0xf6, 0xff, 0xa7, 0xfe, 0xbc, 0x5f, 0xcf, 0xf9, 0x1b, 0x95, 0x57,
  0x6e, 0x51, 0xd2, 0xad, 0x52, 0xf5, 0x47, 0x6d, 0xa5, 0x27, 0x1e, 0xbc,
  0xf7, 0x80, 0xb1, 0x15, 0xa4, 0x89, 0x75, 0x46, 0x5b, 0xab, 0x3c, 0xae,
  0xf9, 0xd6, 0xa4, 0x14, 0xd1, 0xbc, 0x50, 0x3b, 0x8d, 0xaf, 0x50, 0xa6,
  0x23, 0x0c, 0x6c, 0xf1, 0xe7, 0x21, 0x73, 0x88, 0xc3, 0x2a, 0xab, 0xf2,
  0x13, 0xc6, 0x2b, 0x55, 0xea, 0xac, 0x7a, 0xaf, 0x43, 0xc6, 0x30, 0xd4,
  0x94, 0x77, 0x1d, 0x07, 0xec, 0x38, 0x5f, 0x13, 0xe2, 0xf2, 0xbc, 0xfb,
  0x5f, 0x29, 0x2e, 0x87, 0xe4, 0x90, 0xdd, 0xe8, 0xfb, 0x5a, 0x11, 0x4a,
  0x32, 0x23, 0x57, 0x75, 0x06, 0x36, 0x62, 0xba, 0xb2, 0x87, 0x87, 0xbc,
  0xec, 0xd8, 0x93, 0xc5, 0xa4, 0xb7, 0x42, 0xe8, 0xf1, 0x94, 0x80, 0x28,
  0x85, 0x2d, 0xeb, 0x7c, 0x96, 0x22, 0xed, 0xb2, 0x36, 0xa6, 0xfe, 0xab,
  0x36, 0x63, 0x59, 0x4d, 0xf6, 0xbb, 0xaa, 0x64, 0xe5, 0x1c, 0xe9, 0x65,
  0xbb, 0xf5, 0x6e, 0x40, 0x3c, 0xe0, 0x5d, 0xa4, 0xf3, 0x4c, 0xfc, 0xc3,
  0xb0, 0x17, 0x07, 0x10, 0xe8, 0xe6, 0xfa, 0x48, 0xa0, 0x82, 0x70, 0x68,
  0x9c, 0x3f, 0xe7, 0x49, 0x74, 0x7b, 0xa4, 0x51, 0x45, 0x00, 0x79, 0x81,
  0x41, 0x84, 0x72, 0x1b, 0xe3, 0xee, 0x85, 0xa5, 0x0d, 0x11, 0xba, 0x8c,
  0x28, 0x6f, 0x3a, 0xe9, 0xaf, 0x15, 0x4b, 0x5f, 0x53, 0x22, 0xac, 0x7c,
  0x72, 0xeb, 0x7b, 0x4d, 0xa5, 0x5b, 0x43, 0x32, 0x46, 0xb5, 0x52, 0xc1,
  0xb1, 0x5b, 0x34, 0x79, 0x7c, 0x1a, 0xd8, 0x07, 0xc7, 0xb2, 0x24, 0xe7,
  0x2d, 0x90, 0x93, 0x8c, 0x31, 0x35, 0xc9, 0x0b, 0xc2, 0xbc, 0x50, 0x17,
  0xe5, 0x63, 0x1e, 0xe9, 0x29, 0x5f, 0x71, 0x18, 0xb2, 0x74, 0x72, 0x36,
  0x4d, 0xd4, 0x6a, 0xec, 0x40, 0x25, 0xfe, 0x12, 0x03, 0x31, 0x98, 0x9e,
  0x01, 0x4b, 0x6b, 0xce, 0x79, 0x36, 0x19, 0x7d, 0xcb, 0x80, 0xbd, 0xdf,
  0x16, 0xb1, 0xa0, 0xa7, 0xe8, 0x1a, 0x33, 0x16, 0x71, 0xaf, 0xeb, 0x13,
  0x63, 0xfb, 0x64, 0x04, 0x9d, 0xfd, 0x62, 0x66, 0xd3, 0x84, 0x8e, 0x28,
  0x86, 0x7c, 0x1e, 0x07, 0x6f, 0x5b, 0xb0, 0x5e, 0xf1, 0xd8, 0x36, 0x91,
  0x48, 0xb6, 0x21, 0x85, 0x4b, 0xfd, 0xd5, 0x86, 0x02, 0xb9, 0x2e, 0x58,
  0x6c, 0x37, 0xee, 0x36, 0x60, 0x51, 0x04, 0x21, 0xa3, 0x99, 0x1c, 0xe7,
  0x2d, 0xa6, 0x3b, 0xbe, 0x3b, 0x8c, 0xcc, 0x5e, 0x58, 0x75, 0x1f, 0xa0,
  0xec, 0xce, 0xe2, 0xf1, 0xd3, 0xf9, 0x47, 0xe2, 0x74, 0x32, 0xa9, 0xb4,
  0xe1, 0x1a, 0xbf, 0xa7, 0xa2, 0xef, 0xdc, 0x62, 0xe0, 0xc4, 0xd6, 0x7d,
  0x89, 0x7d, 0x64, 0xf9, 0x8d, 0xaf, 0x2f, 0xe4, 0x9f, 0x9b, 0xa4, 0x36,
  0x62, 0xba, 0xda, 0x31, 0xe6, 0xa0, 0x57, 0x9a, 0x49, 0x7a, 0xad, 0x40,
  0x7c, 0x9d, 0x3a, 0x13, 0xa2, 0x37, 0x10, 0x37, 0xd0, 0xe8, 0x27, 0x37,
  0x1c, 0x1d, 0xd0, 0xfa, 0x44, 0x1c, 0x51, 0x45, 0xa7, 0xd3, 0x5b, 0x59,
  0x78, 0x59, 0x55, 0x65, 0xc9, 0xb6, 0x95, 0xb6, 0xec, 0x61, 0x23, 0xa6,
  0x3d, 0xfb, 0x27, 0xe2, 0x73, 0xed, 0x00, 0x38, 0xf0, 0x8e, 0x3a, 0x24,
  0x99, 0x6b, 0x5d, 0xfb, 0xa2, 0x66, 0xcf, 0x30, 0xf6, 0x99, 0x7c, 0x00,
  0x00, 0x00, 0x64, 0x6d, 0x6f, 0x6f, 0x66, 0x00, 0x00, 0x00, 0x10, 0x6d,
  0x66, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00,
  0x00, 0x00, 0x4c, 0x74, 0x72, 0x61, 0x66, 0x00, 0x00, 0x00, 0x1c, 0x74,
  0x66, 0x68, 0x64, 0x00, 0x02, 0x00, 0x38, 0x00, 0x00, 0x00, 0x01, 0x00,
  0x00, 0x03, 0x5a, 0x00, 0x00, 0x01, 0xcd, 0x02, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x14, 0x74, 0x66, 0x64, 0x74, 0x01, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x14, 0x74,
  0x72, 0x75, 0x6e, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
  0x00, 0x00, 0x6c, 0x00, 0x00, 0x01, 0xd5, 0x6d, 0x64, 0x61, 0x74, 0x01,
  0x18, 0x9f, 0xb6, 0xe3, 0x2b, 0xe6, 0x0d, 0xfe, 0x43, 0xf2, 0x7d, 0xdf,
  0x69, 0xf5, 0x3f, 0x27, 0xdc, 0x7e, 0x23, 0x7a, 0x15, 0xf3, 0x9f, 0xab,
  0xf1, 0x82, 0x4e, 0x27, 0x28, 0x58, 0x56, 0x60, 0x2d, 0xd0, 0xf0, 0x17,
  0xf4, 0xe4, 0xfa, 0x48, 0x2b, 0xbd, 0x94, 0x0d, 0xf6, 0x87, 0xf0, 0x1d,
  0xe3, 0x07, 0x90, 0x1d, 0x65, 0x78, 0x3e, 0x64, 0x88, 0x87, 0xd3, 0xdf,
  0x28, 0x0c, 0x63, 0xe2, 0x52, 0xf7, 0x0c, 0x17, 0x33, 0xec, 0x37, 0xe3,
  0xfe, 0x71, 0x06, 0x59, 0x08, 0x54, 0xf7, 0xee, 0x9b, 0x77, 0xfa, 0xfc,
  0xf8, 0xd8, 0x69, 0xd5, 0x3b, 0xcd, 0xc9, 0x9f, 0x5f, 0x18, 0x5c, 0x01,
  0xff, 0x21, 0x55, 0x04, 0x49, 0x28, 0x8e, 0x03, 0x12, 0x4f, 0x1d, 0x32,
  0x67, 0x2f, 0x8d, 0x62, 0x5f, 0xd0, 0xc7, 0xd6, 0x6b, 0xd9, 0x0b, 0xb1,
  0xbe, 0xc3, 0x8a, 0x39, 0xdc, 0x08, 0x48, 0x43, 0x4f, 0x58, 0xbe, 0xc5,
  0xad, 0xe1, 0x4a, 0xb5, 0x6c, 0xf6, 0xfd, 0x91, 0xe7, 0x1c, 0x3a, 0xa1,
  0x91, 0xdc, 0x44, 0x58, 0x26, 0x81, 0x4c, 0x33, 0x8d, 0xb1, 0xe9, 0xc9,
  0xb4, 0xd6, 0x80, 0xe5, 0x81, 0xda, 0x63, 0xdb, 0xc9, 0xdf, 0x11, 0x7b,
  0xfd, 0x5a, 0xdf, 0xea, 0x62, 0x0a, 0x80, 0xfc, 0x79, 0x8e, 0x2f, 0xd4,
  0xbe, 0x77, 0x05, 0x70, 0xdc, 0x7c, 0x6f, 0xbc, 0x77, 0xef, 0xde, 0xeb,
  0x43, 0x11, 0x13, 0x98, 0xf7, 0xe3, 0xf2, 0x5b, 0x3b, 0xff, 0x05, 0x0d,
  0x0a, 0x0b, 0x1c, 0x15, 0x73, 0xcb, 0x71, 0x7a, 0xc7, 0x67, 0x3a, 0xdd,
  0x10, 0xe4, 0xb4, 0xd6, 0x53, 0x72, 0x34, 0x1d, 0xec, 0xb9, 0x2b, 0x60,
  0x68, 0xb8, 0x34, 0xd9, 0x24, 0xd1, 0xf7, 0x02, 0xc6, 0x1a, 0x04, 0xe5,
  0x62, 0x3c, 0xc8, 0x88, 0x0d, 0x40, 0x83, 0x7a, 0xf3, 0xef, 0x12, 0xa8,
  0x7f, 0xbf, 0x53, 0xbd, 0xb7, 0x2b, 0x32, 0xad, 0x19, 0x4f, 0x5b, 0xa2,
  0xb5, 0x02, 0x00, 0x00, 0x48, 0xe9, 0x93, 0xd2, 0x00, 0x01, 0x2a, 0xa8,
  0x91, 0xd2, 0x00, 0x01, 0xdf, 0xf5, 0x28, 0x00, 0x00, 0x30, 0x63, 0xca,
  0xe3, 0x00, 0x00, 0x22, 0x42, 0x56, 0x80, 0x00, 0x00, 0xc7, 0xc6, 0xc1,
  0x50, 0x00, 0x01, 0x2e, 0x10, 0x91, 0x8f, 0x6f, 0x84, 0x8a, 0xca, 0x00,
  0x00, 0x00, 0x01, 0x83, 0x36, 0x83, 0x49, 0x17, 0x3a, 0xd0, 0x58, 0x00,
  0x00, 0x00, 0x07, 0xb1, 0x5a, 0x22, 0xfd, 0x65, 0x9c, 0x90, 0x00, 0x00,
  0x00, 0x02, 0x20, 0x2c, 0xa0, 0x5a, 0x0c, 0x77, 0x69, 0x00, 0x00, 0x00,
  0x00, 0x09, 0x51, 0xc6, 0x93, 0x89, 0x1c, 0x8d, 0x35, 0x92, 0x0c, 0x2a,
  0xc9, 0xe4, 0x55, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x88,
  0x4a, 0x01, 0xf2, 0x0b, 0xa5, 0x45, 0x92, 0x6a, 0xe5, 0x14, 0xcb, 0x27,
  0x24, 0x11, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13, 0x99,
  0x10, 0x8e, 0x2b, 0x01, 0x83, 0x61, 0x2c, 0x6c, 0x11, 0x1a, 0xb6, 0xc8,
  0x39, 0xa4, 0xaa, 0xce, 0xae, 0x68, 0x11, 0xac, 0x42, 0x52, 0x2b, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13, 0x41, 0xc8,
  0xc8, 0x6e, 0x76, 0x89, 0x41, 0xb4, 0x8b, 0x13, 0x50, 0x07, 0x02, 0x2c,
  0x9a, 0x0a, 0x24, 0x1e, 0xec, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x01, 0xc0, 0x00, 0x00, 0x00, 0x69, 0x6d, 0x66, 0x72, 0x61,
  0x00, 0x00, 0x00, 0x51, 0x74, 0x66, 0x72, 0x61, 0x01, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x02, 0xd4, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x7b, 0x01,
  0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x07, 0xeb, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00,
  0x10, 0x6d, 0x66, 0x72, 0x6f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x69
};
static const unsigned int mp4_frag_len = 2701;

static int check_MP4_Frag(vlc_object_t *obj)
{
    stream_t *stream = vlc_stream_MemoryNew(obj, (uint8_t *)mp4_frag, mp4_frag_len, true);
    if(!stream)
        return 1;
    MP4_Box_t *root = MP4_BoxGetRoot(stream);
    vlc_stream_Delete(stream);

    EXPECT(root);

    MP4_BoxDumpStructure(obj, root);

    const MP4_Box_t *box = MP4_BoxGet(root, "moov/trak/mdia/minf/stbl/stsd");
    EXPECT(box);
    EXPECT(box->i_type == ATOM_stsd);

    MP4_BoxFree(root);
    return 0;

failed:
    MP4_BoxFree(root);
    return 1;
}

int main(void)
{
    test_init();

    const char * const args[] = {
        "-vvv",
    };

    libvlc_instance_t *vlc = libvlc_new(ARRAY_SIZE(args), args);
    if(!vlc)
        return 1;

    int ret = check_MP4_BoxGet();

    if( !ret )
      ret = check_MP4_Box(&vlc->p_libvlc_int->obj);

    if( !ret )
      ret = check_MP4_Frag(&vlc->p_libvlc_int->obj);

    libvlc_release(vlc);
    return ret;
}
