#!/usr/bin/env lua

-- This script takes `.svg` file generated by Cairo or Inkscape and
-- replace hardcoded colors so they can be set using CSS or the
-- web browser itself. This makes the accessibility mode work and
-- allows themes to be created for the documentation.

local input, output = ...

if (not input) or (not output) then
    io.stderr:write("_postprocess.lua require <input> and <output>\n")
    io.stderr:write("Got input: "..tostring(input).."\n")
    io.stderr:write("Got output: "..tostring(output).."\n")
    os.exit(0)
end

-- The second 24bit is just the 32 bit converted to #010001 and back.
local FOREGROUNDS = {
    "rgb[(]0[.]5%%,0%%,0[.]5%%[)];",
    "rgb[(]0[.]392157%%,0%%,0[.]392157%%[)];",
    "#010001",
}

local CLASSES = {
    stroke = "svg_stroke",
    fill   = "svg_fill"
}

local i, o = io.open(input, "r"), io.open(output, "w")

if (not i) or (not o) then return end

local line = i:read("*line")
local count

while line do
    -- Deduplicate and concatenate the classes.
    local classes = {}
    local token_found = {}
    local class_str = {}

    for _, token in ipairs { "fill", "stroke" } do

        for _, color in ipairs(FOREGROUNDS) do
            line, count = line:gsub(token .. ":" .. color, token .. ":currentcolor;")

            -- Add the CSS class.
            if count > 0 then
                classes[CLASSES[token]] = true
            end

            line, count = line:gsub(token .. ":" .. color, token .. ":currentcolor;")

            -- Add the CSS class.
            if count > 0 and not token_found[token] then
                token_found[token] = true
				table.insert(classes, CLASSES[token])
            end
        end
    end

    for class in pairs(classes) do
        table.insert(class_str, class)
    end

    if #class_str > 0 then
        line = line:gsub(' style="', ' class="' .. table.concat(class_str, ",") .. '" style="')
    end

    o:write(line .. "\n")

    -- Add the stylesheet.
    if line:sub(1, 6) == "<?xml " then
        o:write('<?xml-stylesheet href="../ldoc.css" type="text/css"?>\n')
    end

    -- Next line.
    line = i:read("*line")
end

o:close()
