/**************************************************************************
 **                                                                      **
 ** Copyright (C) 2011-2025 Lukas Spies                                  **
 ** Contact: https://photoqt.org                                         **
 **                                                                      **
 ** This file is part of PhotoQt.                                        **
 **                                                                      **
 ** PhotoQt is free software: you can redistribute it and/or modify      **
 ** it under the terms of the GNU General Public License as published by **
 ** the Free Software Foundation, either version 2 of the License, or    **
 ** (at your option) any later version.                                  **
 **                                                                      **
 ** PhotoQt is distributed in the hope that it will be useful,           **
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of       **
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        **
 ** GNU General Public License for more details.                         **
 **                                                                      **
 ** You should have received a copy of the GNU General Public License    **
 ** along with PhotoQt. If not, see <http://www.gnu.org/licenses/>.      **
 **                                                                      **
 **************************************************************************/

import QtQuick
import PhotoQt

AnimatedImage {

    id: image

    /*******************************************/
    // these values are READONLY

    property string imageSource: ""
    property real currentScale
    property bool isMainImage
    property Item loaderTop

    /*******************************************/
    // these values are WRITEONLY

    property bool imageMirrorH: false
    property bool imageMirrorV: false

    /*******************************************/

    // this is to explicitely set the sourceSize when the image is ready
    // see below for more details
    signal ensureSourceSizeSet()

    source: (image.imageSource==="" ? "" : ("file:"+PQCScriptsFilesPaths.toPercentEncoding(image.imageSource)))

    asynchronous: true

    // IMPORTANT NOTE REGARDING MIPMAP !!!
    //
    // Whenever the mipmap property changes, then the image is fully reloaded.
    // Making the mipmap property solely depend on the sourceSize makes it switch from false to true most of the time as the
    // sourceSize is reported as QSize(-1,-1) initially. This results, e.g., in the image to flicker when PhotoQt is started with one being passed on.
    // Making the mipmap property by default evaluate to true and ONLY if the image actually is below the threshold in size and the respective setting
    // is switched on, then it is set to false causing the reloading of a very small images (very fast).
    //
    // Another note about using a seperate property to store the sourceSize evaluation for both smooth and mipmap:
    // Such a property is NOT guaranteed to be evaluated before mipmap and might result in it still returning the value of QSize(-1,-1) for
    // the sourceSize even though initialLoad already caused a retrigger of mipmap.
    //
    // This property, initialLoad, ensures the 'true by default' value of mipmap and is set to false once we know the actual sourceSize.
    property bool initialLoad: true

    smooth: !PQCSettings.imageviewInterpolationDisableForSmallImages ||
            (sourceSize.width > PQCConstants.availableWidth && sourceSize.height > PQCConstants.availableHeight) ||
            currentScale > 1.01 || currentScale < 0.95*defaultScale
    mipmap: initialLoad || !PQCSettings.imageviewInterpolationDisableForSmallImages ||
            (sourceSize.width > PQCConstants.availableWidth && sourceSize.height > PQCConstants.availableHeight)

    onStatusChanged: {
        if(status == Image.Ready) {
            // this signal is necessary, otherwise it *can* happen that the image source size is not reported correctly
            // this then results in a 0x0 dimension reported, the fit-to-size does not work, and the image might not show up at all
            ensureSourceSizeSet()
            hasAlpha = PQCScriptsImages.supportsTransparency(image.imageSource)
        } else if(status == Image.Error)
            source = "qrc:/other/errorimage.svg"
    }

    // we use custom mirror properties to be able to animate the mirror process with transforms
    property bool myMirrorH: false
    property bool myMirrorV: false

    // This needs to be set to false!
    // For large (animated) images this leads to excessive memory usage causing all kinds of crashed!
    // These images are typically very quick to load anyways with no noticable improvement when using caching.
    // Also see: https://gitlab.com/lspies/photoqt/-/issues/340
    cache: false

    onMyMirrorHChanged:
        image.imageMirrorH = myMirrorH
    onMyMirrorVChanged:
        image.imageMirrorV = myMirrorV

    property bool hasAlpha: false

    onSourceSizeChanged: {
        image.initialLoad = false
    }

    onPlayingChanged: {
        if(image.visible)
            PQCConstants.animatedImageIsPlaying = playing
    }

    playing: false
    onVisibleChanged: {
        if(image.visible) {
            playing = true
            PQCConstants.animatedImageIsPlaying = true
        }
    }

    Connections {
        target: PQCScriptsShortcuts
        function onSendShortcutMirrorHorizontal() {
            if(image.visible) image.myMirrorH = !image.myMirrorH
        }
        function onSendShortcutMirrorVertical() {
            if(image.visible) image.myMirrorV = !image.myMirrorV
        }
        function onSendShortcutMirrorReset() {
            if(!image.visible) return
            image.myMirrorH = false
            image.myMirrorV = false
        }
    }

    transform: [
        Rotation {
            origin.x: image.width / 2
            origin.y: image.height / 2
            axis { x: 0; y: 1; z: 0 }
            angle: image.myMirrorH ? 180 : 0
            Behavior on angle { enabled: !PQCSettings.generalDisableAllAnimations; NumberAnimation { duration: PQCSettings.imageviewMirrorAnimate ? 200 : 0 } }
        },
        Rotation {
            origin.x: image.width / 2
            origin.y: image.height / 2
            axis { x: 1; y: 0; z: 0 }
            angle: image.myMirrorV ? 180 : 0
            Behavior on angle { enabled: !PQCSettings.generalDisableAllAnimations; NumberAnimation { duration: PQCSettings.imageviewMirrorAnimate ? 200 : 0 } }
        }
    ]

    Image {
        anchors.fill: parent
        z: parent.z-1
        fillMode: Image.Tile

        source: PQCSettings.imageviewTransparencyMarker&&image.hasAlpha ? "/other/checkerboard.png" : ""

    }

    function setMirrorHV(mH : bool, mV : bool) {
        image.myMirrorH = mH
        image.myMirrorV = mV
    }

    Connections {

        target: PQCNotify

        function onPlayPauseAnimationVideo() {

            if(!image.isMainImage)
                return

            image.playing = !image.playing

        }

        function onCurrentAnimatedJump(leftright : int) {
            image.currentFrame = (image.currentFrame+leftright+image.frameCount)%image.frameCount
        }

        function onCurrentAnimatedSaveFrame() {
            PQCScriptsImages.extractFrameAndSave(image.imageSource, image.currentFrame)
        }

    }

    Connections {

        target: PQCConstants

        function onCurrentImageSourceChanged() {
            image.playing = image.isMainImage
        }

    }

}
