//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// G4PVDivisionFactory Implementation file
//
// Author: Ivana Hrivnacova, 04.05.2004  (Ivana.Hrivnacova@cern.ch)
// --------------------------------------------------------------------

#include "G4PVDivisionFactory.hh"
#include "G4PVDivision.hh"
#include "G4VDivisionParameterisation.hh"

//_____________________________________________________________________________

G4PVDivisionFactory* G4PVDivisionFactory::GetInstance()
{
  if (fgInstance == nullptr)
  {
    fgInstance =  new G4PVDivisionFactory;
  }
  return dynamic_cast<G4PVDivisionFactory*>(fgInstance);
} 

//_____________________________________________________________________________

G4VPhysicalVolume* 
G4PVDivisionFactory::CreatePVDivision(const G4String& pName,
                             G4LogicalVolume* pLogical,
                             G4LogicalVolume* pMotherLogical,
                             const EAxis pAxis,
                             const G4int nReplicas,
                             const G4double width,
                             const G4double offset )
{     
  // Create division - with number of divisions and width
  // ---

  return new G4PVDivision(pName, pLogical, pMotherLogical, 
                          pAxis, nReplicas, width, offset);
}    

//_____________________________________________________________________________

G4VPhysicalVolume* 
G4PVDivisionFactory::CreatePVDivision(const G4String& pName,
                             G4LogicalVolume* pLogical,
                             G4LogicalVolume* pMotherLogical,
                             const EAxis pAxis,
                             const G4int nReplicas,
                             const G4double offset )
{     
  // Create division - with number of divisions 
  // ---

  return new G4PVDivision(pName, pLogical, pMotherLogical, 
                          pAxis, nReplicas, offset);
}    

//_____________________________________________________________________________

G4VPhysicalVolume* 
G4PVDivisionFactory::CreatePVDivision(const G4String& pName,
                             G4LogicalVolume* pLogical,
                             G4LogicalVolume* pMotherLogical,
                             const EAxis pAxis,
                             const G4double width,
                             const G4double offset )
{     
  // Create division - with width
  // ---

  return new G4PVDivision(pName, pLogical, pMotherLogical, 
                          pAxis, width, offset);
}    

//_____________________________________________________________________________

G4VPhysicalVolume* 
G4PVDivisionFactory::CreatePVDivision(const G4String& pName,
                             G4LogicalVolume* pLogical,
                             G4LogicalVolume* pMotherLogical,
                             const G4VPVParameterisation* param)
{     
  // Create division - with parameterisation
  // ---

  // Get parameterisation data
  //
  const auto divParam = dynamic_cast<const G4VDivisionParameterisation*>(param);

  if (divParam == nullptr)
  {
    G4Exception("G4PVDivisionFactory::CreatePVDivision()",
                "GeomDiv0001", FatalException,
                "Unexpected parameterisation type!");
    return nullptr;
  }
  
  EAxis axis = divParam->GetAxis();
  G4int nofDivisions = divParam->GetNoDiv();
  G4double width = divParam->GetWidth();
  G4double offset = divParam->GetOffset();

  return new G4PVDivision(pName, pLogical, pMotherLogical, 
                          axis, nofDivisions, width, offset);
}    

//_____________________________________________________________________________

G4bool G4PVDivisionFactory::IsPVDivision(const G4VPhysicalVolume* pv) const
{ 
  // Returns true if pv is division
  // ---

  return dynamic_cast<const G4PVDivision*>(pv) != nullptr;
}

