#![cfg(target_vendor = "apple")]

//! Wrappers around the macOS Security Framework.
#![warn(missing_docs)]
#![allow(non_upper_case_globals)]
#![allow(clippy::manual_non_exhaustive)] // MSRV
#![allow(clippy::bad_bit_mask)] // bitflags
#![allow(clippy::struct_excessive_bools)]
#![allow(clippy::unreadable_literal)]
#![allow(clippy::ignore_without_reason)]

use core_foundation_sys::base::OSStatus;
use security_framework_sys::base::errSecSuccess;

use crate::base::{Error, Result};

#[cfg(test)]
macro_rules! p {
    ($e:expr) => {
        match $e {
            Ok(s) => s,
            Err(e) => panic!("{:?}", e),
        }
    };
}

pub mod access_control;
#[cfg(target_os = "macos")]
pub mod authorization;
pub mod base;
pub mod certificate;
pub mod cipher_suite;
#[cfg(target_os = "macos")]
pub mod cms;
pub mod identity;
pub mod import_export;
pub mod item;
pub mod key;
pub mod os;
pub mod passwords;
#[doc(hidden)]
pub mod passwords_options;
pub mod policy;
pub mod random;
pub mod secure_transport;
pub mod trust;
#[cfg(target_os = "macos")]
pub mod trust_settings;

#[inline(always)]
fn cvt(err: OSStatus) -> Result<()> {
    match err {
        errSecSuccess => Ok(()),
        err => Err(Error::from_code(err)),
    }
}

#[cfg(test)]
mod test {
    use crate::certificate::SecCertificate;

    /// Returns the server certificate (for certificate parsing/identity tests)
    pub fn certificate() -> SecCertificate {
        let certificate = include_bytes!("../test/server.der");
        p!(SecCertificate::from_der(certificate))
    }

    /// Returns the CA certificate (trust anchor for TLS verification)
    pub fn ca_certificate() -> SecCertificate {
        let certificate = include_bytes!("../test/ca.der");
        p!(SecCertificate::from_der(certificate))
    }
}
