﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/apigateway/APIGatewayRequest.h>
#include <aws/apigateway/APIGateway_EXPORTS.h>
#include <aws/apigateway/model/ApiKeySourceType.h>
#include <aws/apigateway/model/EndpointConfiguration.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace APIGateway {
namespace Model {

/**
 * <p>The POST Request to add a new RestApi resource to your
 * collection.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/apigateway-2015-07-09/CreateRestApiRequest">AWS
 * API Reference</a></p>
 */
class CreateRestApiRequest : public APIGatewayRequest {
 public:
  AWS_APIGATEWAY_API CreateRestApiRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateRestApi"; }

  AWS_APIGATEWAY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the RestApi.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateRestApiRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the RestApi.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateRestApiRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A version identifier for the API.</p>
   */
  inline const Aws::String& GetVersion() const { return m_version; }
  inline bool VersionHasBeenSet() const { return m_versionHasBeenSet; }
  template <typename VersionT = Aws::String>
  void SetVersion(VersionT&& value) {
    m_versionHasBeenSet = true;
    m_version = std::forward<VersionT>(value);
  }
  template <typename VersionT = Aws::String>
  CreateRestApiRequest& WithVersion(VersionT&& value) {
    SetVersion(std::forward<VersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the RestApi that you want to clone from.</p>
   */
  inline const Aws::String& GetCloneFrom() const { return m_cloneFrom; }
  inline bool CloneFromHasBeenSet() const { return m_cloneFromHasBeenSet; }
  template <typename CloneFromT = Aws::String>
  void SetCloneFrom(CloneFromT&& value) {
    m_cloneFromHasBeenSet = true;
    m_cloneFrom = std::forward<CloneFromT>(value);
  }
  template <typename CloneFromT = Aws::String>
  CreateRestApiRequest& WithCloneFrom(CloneFromT&& value) {
    SetCloneFrom(std::forward<CloneFromT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of binary media types supported by the RestApi. By default, the
   * RestApi supports only UTF-8-encoded text payloads.</p>
   */
  inline const Aws::Vector<Aws::String>& GetBinaryMediaTypes() const { return m_binaryMediaTypes; }
  inline bool BinaryMediaTypesHasBeenSet() const { return m_binaryMediaTypesHasBeenSet; }
  template <typename BinaryMediaTypesT = Aws::Vector<Aws::String>>
  void SetBinaryMediaTypes(BinaryMediaTypesT&& value) {
    m_binaryMediaTypesHasBeenSet = true;
    m_binaryMediaTypes = std::forward<BinaryMediaTypesT>(value);
  }
  template <typename BinaryMediaTypesT = Aws::Vector<Aws::String>>
  CreateRestApiRequest& WithBinaryMediaTypes(BinaryMediaTypesT&& value) {
    SetBinaryMediaTypes(std::forward<BinaryMediaTypesT>(value));
    return *this;
  }
  template <typename BinaryMediaTypesT = Aws::String>
  CreateRestApiRequest& AddBinaryMediaTypes(BinaryMediaTypesT&& value) {
    m_binaryMediaTypesHasBeenSet = true;
    m_binaryMediaTypes.emplace_back(std::forward<BinaryMediaTypesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A nullable integer that is used to enable compression (with non-negative
   * between 0 and 10485760 (10M) bytes, inclusive) or disable compression (with a
   * null value) on an API. When compression is enabled, compression or decompression
   * is not applied on the payload if the payload size is smaller than this value.
   * Setting it to zero allows compression for any payload size.</p>
   */
  inline int GetMinimumCompressionSize() const { return m_minimumCompressionSize; }
  inline bool MinimumCompressionSizeHasBeenSet() const { return m_minimumCompressionSizeHasBeenSet; }
  inline void SetMinimumCompressionSize(int value) {
    m_minimumCompressionSizeHasBeenSet = true;
    m_minimumCompressionSize = value;
  }
  inline CreateRestApiRequest& WithMinimumCompressionSize(int value) {
    SetMinimumCompressionSize(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The source of the API key for metering requests according to a usage plan.
   * Valid values are: <code>HEADER</code> to read the API key from the
   * <code>X-API-Key</code> header of a request. <code>AUTHORIZER</code> to read the
   * API key from the <code>UsageIdentifierKey</code> from a custom authorizer.</p>
   */
  inline ApiKeySourceType GetApiKeySource() const { return m_apiKeySource; }
  inline bool ApiKeySourceHasBeenSet() const { return m_apiKeySourceHasBeenSet; }
  inline void SetApiKeySource(ApiKeySourceType value) {
    m_apiKeySourceHasBeenSet = true;
    m_apiKeySource = value;
  }
  inline CreateRestApiRequest& WithApiKeySource(ApiKeySourceType value) {
    SetApiKeySource(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The endpoint configuration of this RestApi showing the endpoint types and IP
   * address types of the API. </p>
   */
  inline const EndpointConfiguration& GetEndpointConfiguration() const { return m_endpointConfiguration; }
  inline bool EndpointConfigurationHasBeenSet() const { return m_endpointConfigurationHasBeenSet; }
  template <typename EndpointConfigurationT = EndpointConfiguration>
  void SetEndpointConfiguration(EndpointConfigurationT&& value) {
    m_endpointConfigurationHasBeenSet = true;
    m_endpointConfiguration = std::forward<EndpointConfigurationT>(value);
  }
  template <typename EndpointConfigurationT = EndpointConfiguration>
  CreateRestApiRequest& WithEndpointConfiguration(EndpointConfigurationT&& value) {
    SetEndpointConfiguration(std::forward<EndpointConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A stringified JSON policy document that applies to this RestApi regardless of
   * the caller and Method configuration.</p>
   */
  inline const Aws::String& GetPolicy() const { return m_policy; }
  inline bool PolicyHasBeenSet() const { return m_policyHasBeenSet; }
  template <typename PolicyT = Aws::String>
  void SetPolicy(PolicyT&& value) {
    m_policyHasBeenSet = true;
    m_policy = std::forward<PolicyT>(value);
  }
  template <typename PolicyT = Aws::String>
  CreateRestApiRequest& WithPolicy(PolicyT&& value) {
    SetPolicy(std::forward<PolicyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The key-value map of strings. The valid character set is [a-zA-Z+-=._:/]. The
   * tag key can be up to 128 characters and must not start with <code>aws:</code>.
   * The tag value can be up to 256 characters.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateRestApiRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateRestApiRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether clients can invoke your API by using the default
   * <code>execute-api</code> endpoint. By default, clients can invoke your API with
   * the default <code>https://{api_id}.execute-api.{region}.amazonaws.com</code>
   * endpoint. To require that clients use a custom domain name to invoke your API,
   * disable the default endpoint</p>
   */
  inline bool GetDisableExecuteApiEndpoint() const { return m_disableExecuteApiEndpoint; }
  inline bool DisableExecuteApiEndpointHasBeenSet() const { return m_disableExecuteApiEndpointHasBeenSet; }
  inline void SetDisableExecuteApiEndpoint(bool value) {
    m_disableExecuteApiEndpointHasBeenSet = true;
    m_disableExecuteApiEndpoint = value;
  }
  inline CreateRestApiRequest& WithDisableExecuteApiEndpoint(bool value) {
    SetDisableExecuteApiEndpoint(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_version;
  bool m_versionHasBeenSet = false;

  Aws::String m_cloneFrom;
  bool m_cloneFromHasBeenSet = false;

  Aws::Vector<Aws::String> m_binaryMediaTypes;
  bool m_binaryMediaTypesHasBeenSet = false;

  int m_minimumCompressionSize{0};
  bool m_minimumCompressionSizeHasBeenSet = false;

  ApiKeySourceType m_apiKeySource{ApiKeySourceType::NOT_SET};
  bool m_apiKeySourceHasBeenSet = false;

  EndpointConfiguration m_endpointConfiguration;
  bool m_endpointConfigurationHasBeenSet = false;

  Aws::String m_policy;
  bool m_policyHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;

  bool m_disableExecuteApiEndpoint{false};
  bool m_disableExecuteApiEndpointHasBeenSet = false;
};

}  // namespace Model
}  // namespace APIGateway
}  // namespace Aws
