//
// Syd: rock-solid application kernel
// src/kernel/chmod.rs: chmod(2), fchmod(2), fchmodat(2), and fchmodat2(2) handlers
//
// Copyright (c) 2023, 2024, 2025, 2026 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::AsRawFd;

use libseccomp::ScmpNotifResp;
use nix::{errno::Errno, fcntl::AtFlags, sys::stat::Mode};

use crate::{
    config::PROC_FILE,
    cookie::{safe_fchmod, safe_fchmodat, safe_fchmodat2},
    fd::fd_mode,
    kernel::{syscall_path_handler, to_atflags, to_mode},
    lookup::FsFlags,
    path::XPathBuf,
    req::{PathArgs, SysArg, UNotifyEventRequest},
    sandbox::SandboxGuard,
};

pub(crate) fn sys_fchmod(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Strip undefined/invalid mode bits.
    let mode = to_mode(req.data.args[1]);

    let argv = &[SysArg {
        dirfd: Some(0),
        fsflags: FsFlags::MUST_PATH,
        ..Default::default()
    }];
    syscall_path_handler(request, "fchmod", argv, |path_args, request, sandbox| {
        // SAFETY:
        // 1. SysArg has one element.
        // 2. We use MUST_PATH, dir refers to the file.
        #[expect(clippy::disallowed_methods)]
        let path = path_args.0.as_ref().unwrap();
        let fd = path.dir();

        // SAFETY:
        // 1. We apply force_umask to chmod(2) modes to ensure consistency.
        // 2. Umask is only forced for regular files.
        // 3. Sticky bit is immutable for directories unless trace/allow_unsafe_sticky:1 is set.
        let mut mode = mode;
        if path.is_file() {
            let umask = sandbox.umask.unwrap_or(Mode::empty());
            mode &= !umask;
        } else if !sandbox.flags.allow_unsafe_sticky()
            && path.is_dir()
            && fd_mode(fd)?.contains(Mode::S_ISVTX)
        {
            mode.insert(Mode::S_ISVTX);
        }
        drop(sandbox); // release the read-lock.

        safe_fchmod(fd, mode).map(|_| request.return_syscall(0))
    })
}

pub(crate) fn sys_chmod(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Strip undefined/invalid mode bits.
    let mode = to_mode(req.data.args[1]);

    let argv = &[SysArg {
        path: Some(0),
        ..Default::default()
    }];

    syscall_path_handler(request, "chmod", argv, |path_args, request, sandbox| {
        syscall_chmod_handler(request, sandbox, path_args, mode)
    })
}

pub(crate) fn sys_fchmodat(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Strip undefined/invalid mode bits.
    let mode = to_mode(req.data.args[2]);

    // Note: Unlike fchmodat2, fchmodat always resolves symbolic links.
    let argv = &[SysArg {
        dirfd: Some(0),
        path: Some(1),
        ..Default::default()
    }];

    syscall_path_handler(request, "fchmodat", argv, |path_args, request, sandbox| {
        syscall_chmod_handler(request, sandbox, path_args, mode)
    })
}

pub(crate) fn sys_fchmodat2(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Reject undefined/invalid/unused flags.
    let flags = match to_atflags(req.data.args[3], AtFlags::AT_SYMLINK_NOFOLLOW) {
        Ok(flags) => flags,
        Err(errno) => return request.fail_syscall(errno),
    };

    // SAFETY: Strip undefined/invalid mode bits.
    let mode = to_mode(req.data.args[2]);

    let mut fsflags = FsFlags::MUST_PATH;
    if flags.contains(AtFlags::AT_SYMLINK_NOFOLLOW) {
        fsflags |= FsFlags::NO_FOLLOW_LAST
    }

    let argv = &[SysArg {
        dirfd: Some(0),
        path: Some(1),
        fsflags,
        ..Default::default()
    }];

    syscall_path_handler(request, "fchmodat2", argv, |path_args, request, sandbox| {
        syscall_chmod_handler(request, sandbox, path_args, mode)
    })
}

/// A helper function to handle chmod, fchmodat, and fchmodat2 syscalls.
fn syscall_chmod_handler(
    request: &UNotifyEventRequest,
    sandbox: SandboxGuard,
    args: PathArgs,
    mut mode: Mode,
) -> Result<ScmpNotifResp, Errno> {
    // SAFETY:
    // 1. SysArg has one element.
    // 2. We use MUST_PATH, dir refers to the file.
    #[expect(clippy::disallowed_methods)]
    let path = args.0.as_ref().unwrap();
    let fd = path.dir();

    // SAFETY:
    // 1. We apply force_umask to chmod modes to ensure consistency.
    // 2. Umask is only forced for regular files.
    // 3. Sticky bit is immutable for directories unless trace/allow_unsafe_sticky:1 is set.
    if path.is_file() {
        let umask = sandbox.umask.unwrap_or(Mode::empty());
        mode &= !umask;
    } else if !sandbox.flags.allow_unsafe_sticky()
        && path.is_dir()
        && fd_mode(fd)?.contains(Mode::S_ISVTX)
    {
        mode.insert(Mode::S_ISVTX);
    }
    drop(sandbox); // release the read-lock.

    match safe_fchmodat2(fd, mode) {
        Ok(_) => Ok(()),
        Err(Errno::ENOSYS) => {
            // Fallback to `/proc` indirection,
            //
            // path to fd is open already!
            let pfd = XPathBuf::from_self_fd(fd.as_raw_fd())?;
            safe_fchmodat(PROC_FILE(), &pfd, mode)
        }
        Err(errno) => Err(errno),
    }
    .map(|_| request.return_syscall(0))
}
